#ifndef __CScrollBarImages__
#define __CScrollBarImages__

#include "../Graphics/IImage.hpp"
#include "../Graphics/CGraphics.hpp"

//	===========================================================================

using Exponent::GUI::Graphics::CGraphics;
using Exponent::GUI::Graphics::IImage;

//	===========================================================================

namespace Exponent
{
	namespace GUI
	{
		namespace Controls
		{
			/**
			 * @class CScrollBarImages CScrollBarImages.hpp
			 * @brief Images required by the scroll bars
			 *
			 * Scroll bars are made up of five distinct image parts\n
			 * \n
			 * |--------------|\n
			 * |    TOP CAP   |\n
			 * |--------------|\n
			 * |              |\n
			 * |              |\n
			 * |    REPEAT    |\n
			 * |              |\n
			 * |              |\n
			 * |--------------|\n
			 * |    THUMB     |\n
			 * |--------------|\n
			 * |              |\n
			 * |              |\n
			 * |    REPEAT    |\n
			 * |              |\n
			 * |              |\n
			 * |--------------|\n
			 * |  BOTTOM CAP  |\n
			 * |--------------|\n
			 * \n
			 * The Top Cap and Bottom Cap and Thumb are single images and are simply drawn im place as per any normal image\n
			 * The repeat needs to be a small tileable graphic that will be repeated accross the remaining area to fill in the\n
			 * gaps between top bottom and thumb.
			 *
			 * @date 21/04/2005
			 * @author Paul Chana
			 * @version 1.0.0 Initial version
			 * @version 1.0.1 Update to suit new CScrollPanel
			 *
			 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
			 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
			 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
			 * All content is the Intellectual property of Exp Digital Uk.\n
			 * Certain sections of this code may come from other sources. They are credited where applicable.\n
			 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
			 *
			 * $Id: CScrollBarImages.hpp,v 1.3 2007/02/08 21:07:54 paul Exp $
			 */
			class CScrollBarImages : public CCountedObject
			{
				/** @cond */
				EXPONENT_CLASS_DECLARATION;
				/** @endcond */

//	===========================================================================

			public:

//	===========================================================================

				/**
				 * Construction
				 */
				CScrollBarImages();

				/**
				 * Destruction
				 */
				virtual ~CScrollBarImages();

//	===========================================================================

				/**
				 * Copy operator
				 */
				CScrollBarImages &operator = (const CScrollBarImages &other);

//	===========================================================================

				/**
				 * Check if all the images are loaded
				 * @retval bool True if all images are valid
				 */
				bool isValid();

//	===========================================================================

				/**
				 * Draw the scroll bar images
				 * @param graphics The graphics handle
				 * @param area The area to draw them in
				 * @param isVertical True if scroll bar is vertical, false otherwise
				 * @retval bool True if drawn properly, false otherwise
				 */
				bool drawScrollBarImages(CGraphics &graphics, const CRect &area, const bool isVertical);

//	===========================================================================

				/**
				 * Set the images
				 * @param topCap The top cap
				 * @param bottomCap The bottom cap
				 * @param thumb The central thumb image
				 * @param repeat The repeating part of the image
				 */
				void setImages(IImage *topCap, IImage *bottomCap, IImage *thumb, IImage *repeat);

				/**
				 * Get the top image
				 * @retval IImage* THe top cap
				 */
				IImage *getTopImage() const { return m_topCap; }

				/**
				 * Get the bottom image
				 * @retval IImage* The bottom image
				 */
				IImage *getBottomImage() const { return m_bottomCap; }

				/**
				 * Get the thumb image
				 * @retval IImage* The thumb (central) part
				 */
				IImage *getThumbImage() const { return m_thumb; }

				/**
				 * Get the repeat image
				 * @retval IImage* The repeating part of the scroll bar
				 */
				IImage *getRepeatImage() const { return m_repeat; }

//	===========================================================================

				/**
				 * Load from resource folder
				 * @param resourceFolder The path to the folder to load from, expects to find\n
				 * Targa images named\n
				 * ScrollBarTopCap\n
				 * ScrollBarRepeat\n
				 * ScrollBarThumb\n
				 * ScrollBarEndCap
				 */
				void loadFromResourceFolder(const CSystemString &resourceFolder);

//	===========================================================================

			protected:

//	===========================================================================

				IImage *m_topCap;					/**< End cap - top / left */
				IImage *m_bottomCap;				/**< End cap - bottom / right */
				IImage *m_thumb;					/**< The thumb track image */
				IImage *m_repeat;					/**< The repeat image */
				bool m_isValid;						/**< All images loaded? */
			};
		}
	}
}
#endif	// End of CScrollBarImages.hpp